<?php
session_start();
require_once '../config/database.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Non autorisé']);
    exit();
}

$current_user_id = (int)$_SESSION['user_id'];
$database = new Database();
$db = $database->getConnection();

$action = $_POST['action'] ?? $_GET['action'] ?? null;
$target_user_id = isset($_POST['target_user_id']) ? (int)$_POST['target_user_id'] : (isset($_GET['target_user_id']) ? (int)$_GET['target_user_id'] : null);

if (!$action) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Action non spécifiée.']);
    exit();
}

// Determine user_one_id (smaller) and user_two_id (larger)
$user_one_id = null;
$user_two_id = null;
if ($target_user_id) {
    if ($current_user_id == $target_user_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Vous не pouvez pas effectuer cette action avec vous-même.']);
        exit();
    }
    $user_one_id = min($current_user_id, $target_user_id);
    $user_two_id = max($current_user_id, $target_user_id);
}


try {
    switch ($action) {
        case 'send_request':
            if (!$target_user_id) throw new Exception("ID utilisateur cible manquant.");

            // Check if a relationship already exists
            $check_stmt = $db->prepare("SELECT status FROM friends WHERE user_one_id = :u1 AND user_two_id = :u2");
            $check_stmt->bindParam(':u1', $user_one_id);
            $check_stmt->bindParam(':u2', $user_two_id);
            $check_stmt->execute();
            $existing_friendship = $check_stmt->fetch(PDO::FETCH_ASSOC);

            if ($existing_friendship) {
                if ($existing_friendship['status'] === 'accepted') {
                    throw new Exception("Vous êtes déjà amis.");
                } else { // pending
                    throw new Exception("Une demande d'ami est déjà en attente.");
                }
            }

            $stmt = $db->prepare("INSERT INTO friends (user_one_id, user_two_id, status, action_user_id) VALUES (:u1, :u2, 'pending', :action_by)");
            $stmt->bindParam(':u1', $user_one_id);
            $stmt->bindParam(':u2', $user_two_id);
            $stmt->bindParam(':action_by', $current_user_id); // Current user sends the request
            $stmt->execute();
            echo json_encode(['success' => true, 'message' => 'Demande d\'ami envoyée.']);
            break;

        case 'accept_request':
            if (!$target_user_id) throw new Exception("ID utilisateur cible manquant.");
            
            // The request was sent by target_user_id (action_user_id) to current_user_id
            $stmt = $db->prepare("UPDATE friends SET status = 'accepted', action_user_id = :current_user_id 
                                  WHERE user_one_id = :u1 AND user_two_id = :u2 AND status = 'pending' AND action_user_id = :target_user_id");
            $stmt->bindParam(':current_user_id', $current_user_id);
            $stmt->bindParam(':u1', $user_one_id);
            $stmt->bindParam(':u2', $user_two_id);
            $stmt->bindParam(':target_user_id', $target_user_id); // The one who sent the request
            $stmt->execute();

            if ($stmt->rowCount() > 0) {
                echo json_encode(['success' => true, 'message' => 'Demande d\'ami acceptée.']);
            } else {
                throw new Exception("Demande d'ami non trouvée ou déjà traitée.");
            }
            break;

        case 'decline_request':
        case 'remove_friend': // Same logic: delete the row
            if (!$target_user_id) throw new Exception("ID utilisateur cible manquant.");

            $stmt = $db->prepare("DELETE FROM friends WHERE user_one_id = :u1 AND user_two_id = :u2");
            // For decline, we might want to check if action_user_id was the target_user_id and status was pending.
            // For remove, status should be 'accepted'.
            // The simple delete works for both if the JS sends correct requests.
            $stmt->bindParam(':u1', $user_one_id);
            $stmt->bindParam(':u2', $user_two_id);
            $stmt->execute();

            if ($stmt->rowCount() > 0) {
                $message = ($action === 'decline_request') ? 'Demande d\'ami refusée.' : 'Ami retiré.';
                echo json_encode(['success' => true, 'message' => $message]);
            } else {
                throw new Exception("Relation non trouvée.");
            }
            break;

        case 'list_friends':
            $stmt = $db->prepare("
                SELECT 
                    CASE
                        WHEN f.user_one_id = :current_user_id THEN f.user_two_id
                        ELSE f.user_one_id
                    END as friend_user_id,
                    u.display_name, u.avatar_url
                FROM friends f
                JOIN users u ON u.id = (CASE WHEN f.user_one_id = :current_user_id THEN f.user_two_id ELSE f.user_one_id END)
                WHERE (f.user_one_id = :current_user_id OR f.user_two_id = :current_user_id) AND f.status = 'accepted'
                ORDER BY u.display_name ASC
            ");
            $stmt->bindParam(':current_user_id', $current_user_id);
            $stmt->execute();
            $friends = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo json_encode(['success' => true, 'data' => $friends]);
            break;

        case 'list_pending_requests': // Incoming requests for current user
            $stmt = $db->prepare("
                SELECT 
                    f.action_user_id as friend_user_id, -- The user who sent the request
                    u.display_name, u.avatar_url
                FROM friends f
                JOIN users u ON u.id = f.action_user_id
                WHERE ((f.user_one_id = :current_user_id AND f.action_user_id = f.user_two_id) OR (f.user_two_id = :current_user_id AND f.action_user_id = f.user_one_id))
                AND f.status = 'pending'
                ORDER BY f.created_at DESC
            ");
            $stmt->bindParam(':current_user_id', $current_user_id);
            $stmt->execute();
            $pending_requests = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo json_encode(['success' => true, 'data' => $pending_requests]);
            break;

        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Action invalide.']);
            break;
    }

} catch (PDOException $e) {
    http_response_code(500);
    error_log("Friends API Error (PDO): " . $e->getMessage() . " for action " . $action);
    echo json_encode(['success' => false, 'error' => 'Erreur base de données.', 'details' => $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(400); // Bad request for logical errors
    error_log("Friends API Error (Logic): " . $e->getMessage() . " for action " . $action);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
?>
